#!/bin/sh
# Syntax: $0 pid uid gid signalId timeInSecSinceEpoch coreFileSizeLimit comm

logger "Crash dump: $@"
 
PID=$1
SIGNAL=$4
EPOCH_SECS=$5
COMM=$7
 
CRASH_EXE=$(readlink /proc/${PID}/exe)

if [ "$?" != "0" ]; then
    logger "ERROR: Can't readlink on PID $PID"
    exit 1
fi

PROCNAME=$(basename ${CRASH_EXE})

if [ "$?" != "0" ]; then
    logger "ERROR: Can't get basename on exe $CRASH_EXE"
    exit 1
fi

DESTDIR=/var/log
BASENAME=core.${PROCNAME}__${COMM}__${SIGNAL}__${EPOCH_SECS}
# Important: we need to replace '-' with '_'. Systemd does some escaping on '-' and replaces them 
# with '/' when we use the %I variable, which breaks the path of the tar file. To avoid problems in
# this script, we also remove all non-alphanumeric characters, except for spaces and slashes, which 
# we also replace with '_'. Dots are also preserved.
BASENAME=$(echo $BASENAME | tr -dc '[:alnum:]\n\r_. /-' | tr ' /-' '_')
TARGETFILE=${BASENAME}.lz4
TARGETPATH=${DESTDIR}/${TARGETFILE}

PROCDUMPDIR=${BASENAME}_ProcessDump
PROCDUMPPATH=${DESTDIR}/${PROCDUMPDIR}

LOGFILE=${BASENAME}.log
LOGPATH=${DESTDIR}/${LOGFILE}

TARPATH=${DESTDIR}/${BASENAME}.tar

function Cleanup                                                            
{
    logger "Caught signal while generating coredump, cleanup and exit."
    rm -f ${TARGETPATH}
    rm -rf ${PROCDUMPPATH}
    rm -f ${LOGPATH}
    rm -f ${TARPATH}
    exit 1
}

# Install trap handler to cleanup generated files when catching a signal.                            
trap Cleanup 1 2 3 4 5 6 7 8 11 13 14 15

# Generate ProcessDump report for the crashed PID. 
# This contains useful information such as thread names and memory map file.
if [ -f /usr/local/bin/ProcessDump ]; then
    /usr/local/bin/ProcessDump ${DESTDIR} --pid ${PID} --reportdir ${PROCDUMPDIR}
fi

# Store information in a log file to allow to associate the dump with a time/date and a firmware version.
echo "timedatectl:" > ${LOGPATH}
echo "$(timedatectl)" >> ${LOGPATH}
echo >> ${LOGPATH}
echo "info3.gg:" >> ${LOGPATH}
cat /rw/firmware/info3.gg >> ${LOGPATH} 2>&1

FAIL=false

# Compress the core dump with lz4
/usr/bin/lz4 > ${TARGETPATH}

if [ "$?" != "0" ]; then
    FAIL=true
fi

if [ "$FAIL" = false ]; then
    # Do not fail the generation of the tar if for some reason one of its
    # contents could not be generated.
    test -e "${TARGETPATH}"   || TARGETFILE=""
    test -e "${PROCDUMPPATH}" || PROCDUMPDIR=""
    test -e "${LOGPATH}"      || LOGFILE=""

    # Package the compressed dump, the ProcessDump report and the log file in a tar
    tar -cf ${TARPATH} -C ${DESTDIR} ${TARGETFILE} ${PROCDUMPDIR} ${LOGFILE}

    if [ "$?" != "0" ]; then
        rm -f ${TARPATH}
        FAIL=true
    fi
fi

# Cleanup temporary files that are now packaged in the tar.
rm -f ${TARGETPATH}
rm -rf ${PROCDUMPPATH}
rm -f ${LOGPATH}

if [ "$FAIL" = false ]; then
    # Invoke systemd-coredump service with the tar file as "%I" variable.
    systemctl start systemd-coredump@${TARPATH}.service
    if [ "$?" = "0" ]; then
        exit 0
    else
        rm -f ${TARPATH}
        logger "ERROR: Failed to start systemd-coredump@.service."
        exit 1
    fi
else
    logger "ERROR: Failed to generate coredump."
    exit 1
fi

